<?php
/**
 * Plugin Name: EDD Social Proof Notifications
 * Description: Displays real-time purchase notifications for Easy Digital Downloads.
 * Version: 1.0
 * Author: Asheville Web Design
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class EDD_Social_Proof {
    public function __construct() {
        add_action('wp_footer', [$this, 'enqueue_scripts']);
        add_action('wp_ajax_nopriv_get_recent_edd_purchases', [$this, 'get_recent_purchases']);
        add_action('wp_ajax_get_recent_edd_purchases', [$this, 'get_recent_purchases']);
        add_action('admin_menu', [$this, 'create_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
    }

    public function enqueue_scripts() {
        if (!is_checkout()) {
            wp_enqueue_script('edd-social-proof', plugin_dir_url(__FILE__) . 'social-proof.js', ['jquery'], '1.0', true);
            wp_localize_script('edd-social-proof', 'EDD_Social_Proof_Ajax', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'display_time' => get_option('edd_social_proof_display_time', 5),
                'notification_delay' => get_option('edd_social_proof_delay', 3),
                'bg_color' => get_option('edd_social_proof_bg_color', '#333'),
                'text_color' => get_option('edd_social_proof_text_color', '#fff')
            ]);
            wp_enqueue_style('edd-social-proof-style', plugin_dir_url(__FILE__) . 'social-proof.css');
        }
    }

    public function get_recent_purchases() {
        global $wpdb;
        
        $results = $wpdb->get_results("SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = '_edd_payment_meta' ORDER BY post_id DESC LIMIT 5");
        
        $notifications = [];
        foreach ($results as $result) {
            $data = maybe_unserialize($result->meta_value);
            if (!empty($data['email'])) {
                $notifications[] = [
                    'name' => explode('@', $data['email'])[0],
                    'product' => $data['downloads'][0]['name']
                ];
            }
        }
        
        wp_send_json($notifications);
    }

    public function create_settings_page() {
        add_options_page('EDD Social Proof', 'EDD Social Proof', 'manage_options', 'edd-social-proof', [$this, 'settings_page_content']);
    }

    public function register_settings() {
        register_setting('edd_social_proof_settings', 'edd_social_proof_display_time');
        register_setting('edd_social_proof_settings', 'edd_social_proof_delay');
        register_setting('edd_social_proof_settings', 'edd_social_proof_bg_color');
        register_setting('edd_social_proof_settings', 'edd_social_proof_text_color');
    }

    public function settings_page_content() {
        ?>
        <div class="wrap">
            <h1>EDD Social Proof Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('edd_social_proof_settings'); ?>
                <?php do_settings_sections('edd_social_proof_settings'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="edd_social_proof_display_time">Display Time (seconds)</label></th>
                        <td><input type="number" name="edd_social_proof_display_time" value="<?php echo get_option('edd_social_proof_display_time', 5); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="edd_social_proof_delay">Notification Delay (seconds)</label></th>
                        <td><input type="number" name="edd_social_proof_delay" value="<?php echo get_option('edd_social_proof_delay', 3); ?>" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="edd_social_proof_bg_color">Notification Background Color</label></th>
                        <td><input type="text" name="edd_social_proof_bg_color" value="<?php echo get_option('edd_social_proof_bg_color', '#333'); ?>" class="color-picker" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="edd_social_proof_text_color">Notification Text Color</label></th>
                        <td><input type="text" name="edd_social_proof_text_color" value="<?php echo get_option('edd_social_proof_text_color', '#fff'); ?>" class="color-picker" /></td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}

new EDD_Social_Proof();
